// msn.cpp 
// Copyright (c) 2002 - 2009 Ivan Mendez Licensed under GNU GPL
// For more information, see License.txt

#include "msn.h"
#include "msgr_e.h"
#include "MessengerEvents.h"

IMSNMsgrObject2 *MSNMsgr;

// If COM is already initialized this should be true
bool COMInitialized = false;

IMessenger3 *msn;	// IMessenger object
// Release when no longer needed
IMessengerContact *IMcontact; 
IMessengerGroup *IMgroup; 
IMessengerWindow *IMwindow;
IMessengerContacts *pContacts;
IMessengerGroups *pGroups;
// Used to send messages to mIRC
HWND mirc_hwnd;		
HANDLE hFileMap;
LPSTR mData;

extern Host *pHost;

//----------------------------------------------------------------------------
// bstr_to_char -- Converts a BSTR string to a char string
//
// Parameters
//	string	-- Pointer to the character string to convert
//	buffer	-- Pointer to a buffer that receives the converted string
//	size	-- Size of the buffer in bytes
// 
// Returns
//	The size of converted string
//----------------------------------------------------------------------------

size_t bstr_to_char(const BSTR string, char *buffer, int size)
{
	ZeroMemory(buffer,size);
	UINT len = SysStringLen(string);
	WideCharToMultiByte(CP_UTF8,0,string,len,buffer,size,NULL,NULL);
	return len;
}

//----------------------------------------------------------------------------
// char_to_bstr -- Converts a char string to a BSTR string  
//
// Parameters
//	string	-- Pointer to the character string to convert
//	buffer	-- Pointer to a buffer that receives the converted string
// 
// Notes
//	The buffer must be freed with SysFreeString after use
//----------------------------------------------------------------------------

void char_to_bstr(const char *string, BSTR& buffer)
{
	int len = lstrlen(string);
	int sz = MultiByteToWideChar(CP_UTF8, 0, string, -1, 0, 0);
	WCHAR *tmp = new WCHAR[sz+1];
	MultiByteToWideChar(CP_UTF8, 0, string, -1, tmp, sz);
	tmp[sz] = '\0';
	buffer = SysAllocString(tmp);
	delete[] tmp;
}


//----------------------------------------------------------------------------
// Result -- Returns the result a string
//
// Parameters
//	res		-- The result of a MessengerAPI function
//----------------------------------------------------------------------------

char * Result(HRESULT res) 
{
    switch(res) {
	case S_OK:
        return "S_OK";
        break;
    case E_OUTOFMEMORY:
        return "E_OUTOFMEMORY";
        break;
    case E_INVALIDARG:
        return "E_INVALIDARG";
        break;
    case S_FALSE:
        return "S_FALSE";
        break;
    case E_NOTIMPL:
        return "E_NOTIMPL";
        break;
    case E_FAIL:
        return "E_FAIL";
        break;
    case MSGR_E_AUDIO_UI_ACTIVE:
        return "MSGR_E_AUDIO_UI_ACTIVE";
        break;
    case MSGR_E_OPTION_UI_ACTIVE:
        return "MSGR_E_OPTION_UI_ACTIVE";
        break;
    case MSGR_E_NOT_LOGGED_ON:
        return "MSGR_E_NOT_LOGGED_ON";
        break;
    case MSGR_E_LOGON_UI_ACTIVE:
        return "MSGR_E_LOGON_UI_ACTIVE";
        break;
    case MSGR_E_ALREADY_LOGGED_ON:
        return "MSGR_E_ALREADY_LOGGED_ON";
        break;
    case RPC_X_NULL_REF_POINTER:
        return "RPC_X_NULL_REF_POINTER";
        break;
    case MSGR_E_CONTACT_UI_ACTIVE:
        return "MSGR_E_CONTACT_UI_ACTIVE";
        break;
    case MSGR_E_PAGING_UNAVAILABLE:
        return "MSGR_E_PAGING_UNAVAILABLE";
        break;
    case MSGR_E_PHONESERVICE_UNAVAILABLE:
        return "MSGR_E_PHONESERVICE_UNAVAILABLE";
        break;
    case MSGR_E_PHONE_INVALID_NUMBER:
        return "MSGR_E_PHONE_INVALID_NUMBER";
        break;
    case DISP_E_BADVARTYPE:
        return "DISP_E_BADVARTYPE";
        break;
    case MSGR_E_USER_NOT_FOUND:
        return "MSGR_E_USER_NOT_FOUND";
        break;
    case MSGR_E_GROUPS_NOT_ENABLED:
        return "MSGR_E_GROUPS_NOT_ENABLED";
        break;
    case MSGR_E_GROUP_NAME_TOO_LONG:
        return "MSGR_E_GROUP_NAME_TOO_LONG";
        break;
    case MSGR_E_BAD_GROUP_NAME:
        return "MSGR_E_BAD_GROUP_NAME";
        break;
    case E_POINTER:
        return "E_POINTER";
        break;
    default:
        return "E_FAIL";
    }
}

//----------------------------------------------------------------------------
// Result -- Returns the status text
//
// Parameters
//	pmStatus	-- The status 
//----------------------------------------------------------------------------

char * Status(MISTATUS pmStatus) 
{
    switch (pmStatus) {
		case MISTATUS_OFFLINE:
			return "Offline";
			break;
		case MISTATUS_ONLINE:
			return "Online";
			break;
		case MISTATUS_INVISIBLE:
			return "Invisible";
			break;
		case MISTATUS_BUSY:
			return "Busy";
			break;
		case MISTATUS_BE_RIGHT_BACK:
			return "Be right back";
			break;
		case MISTATUS_IDLE:
			return "Idle";
			break;
		case MISTATUS_AWAY:
			return "Away";
			break;
		case MISTATUS_ON_THE_PHONE:
			return "On the phone";
			break;
		case MISTATUS_OUT_TO_LUNCH:
			return "Out to lunch";
			break;
		case MISTATUS_LOCAL_FINDING_SERVER:
			return "Finding server";
			break;
		case MISTATUS_LOCAL_SYNCHRONIZING_WITH_SERVER:
			return "Sincronizing with server";
			break;
		case MISTATUS_LOCAL_CONNECTING_TO_SERVER:
			return "Connecting to server";
			break;
		case MISTATUS_LOCAL_DISCONNECTING_FROM_SERVER:
			return "Disconnecting from server";
			break;
		case MISTATUS_UNKNOWN:
		default:
			return "Unknown";
    }
}

//----------------------------------------------------------------------------
// GetContact -- Creates an IMessengerContact object from a SiginName
//
// Parameters
//	bstrSigninName	-- The contact's SigninName
//
// Returns
//	S_OK on success 
//----------------------------------------------------------------------------

HRESULT GetContact(BSTR bstrSigninName) {
    BSTR bstrServiceID;
    IDispatch *ppMContact = NULL;
    HRESULT res = msn->get_MyServiceId(&bstrServiceID);
    if(FAILED(res))
        return res; 
    res = msn->raw_GetContact((BSTR)bstrSigninName,bstrServiceID,&ppMContact);
    if(FAILED(res))
        return res; 
    res = ppMContact->QueryInterface(__uuidof(IMessengerContact),reinterpret_cast<void **>(&IMcontact));
    if (ppMContact) ppMContact->Release();
    return res;
}

//----------------------------------------------------------------------------
// getword -- Gets a word from a string
//
// Parameters
//	p	-- Pointer to the string
//	c	-- Character that separates words if not specified space is used
//
// Returns
//	A pointer to the word
//----------------------------------------------------------------------------

char* getword(char*& p, int c)
{
    if (!p || !lstrlen(p))
        return p;
    while (*p == c)
        p++;
    char* r = p;
    while (*p && (*p != c))
        p++;
    if (*p)
    {
        *p++ = 0;
        while (*p == c)
            p++;
    }
    return r;
}

//----------------------------------------------------------------------------
// Signal -- Sends a signal to mIRC
//
// Parameters
//	s	-- Parameter for signal
//----------------------------------------------------------------------------

void Signal(char* s) {
	_snprintf_s(mData,900,_TRUNCATE,"/.signal msn.%s",s);
    SendMessage(mirc_hwnd, WM_USER + 200,0,0);
    return;
}

//----------------------------------------------------------------------------
// get_MyContacts -- Gets an IMessengerContacts interface
//----------------------------------------------------------------------------

HRESULT get_MyContacts() {
    IDispatch *ppMContacts = NULL;
    HRESULT res = msn->get_MyContacts(&ppMContacts);
    if(FAILED(res))
        return res;
    res = ppMContacts->QueryInterface(__uuidof(IMessengerContacts),(void**)&pContacts);
    ppMContacts->Release();
    return res;
}

//----------------------------------------------------------------------------
// get_MyGroups -- Gets an IMessengerGroups interface
//----------------------------------------------------------------------------

HRESULT get_MyGroups() {
    IDispatch *ppGroups = NULL;
    HRESULT res = msn->get_MyGroups(&ppGroups);
    if(FAILED(res))
        return res;
    res = ppGroups->QueryInterface(__uuidof(IMessengerGroups),(void**)&pGroups);
    ppGroups->Release();
    return res;
}

extern "C" { 

//----------------------------------------------------------------------------
// LoadDll -- Initializes the objects and handles used by the dll
//----------------------------------------------------------------------------

	void WINAPI LoadDll(LOADINFO* li)	
	{

		//Initialize the COM library
		HRESULT hRes = CoInitializeEx(NULL,COINIT_MULTITHREADED);
		if (FAILED(hRes))
		{
			// if COM is loaded
			if (hRes == RPC_E_CHANGED_MODE)
				COMInitialized = true;
			else {
				li->mKeep = 0;
				return;
			}
		}
		
		//Create the msn Object
		hRes = CoCreateInstance(CLSID_Messenger, NULL, CLSCTX_ALL, 
			IID_IMessenger3, reinterpret_cast<void **>(&msn));
		if (FAILED(hRes))
		{
			li->mKeep = 0;
			if (!COMInitialized) // Only unload COM if is loaded by the dll
				CoUninitialize();
			return;
		}
		hRes = CoCreateInstance(CLSID_MSNMsgrObject,0,CLSCTX_LOCAL_SERVER,
			__uuidof(IMSNMsgrObject2),reinterpret_cast<void **>(&MSNMsgr));
		if (FAILED(hRes))
		{
			li->mKeep = 0;
			if (msn) msn->Release();
			if (!COMInitialized) // Only unload COM if is loaded by the dll
				CoUninitialize();
			return;
		}
		pHost = new Host;

		//Create a mapped file to send messages to mIRC
		hFileMap = CreateFileMapping(INVALID_HANDLE_VALUE,0,
			PAGE_READWRITE,0,4096,"mIRC");
		if (hFileMap == NULL)
		{
			li->mKeep = 0;
			if (msn) msn->Release();
			if (MSNMsgr) MSNMsgr->Release();
			if (pHost) pHost->Release();
			if (!COMInitialized) // Only unload COM if is loaded by the dll
				CoUninitialize();
			return;
		}

		mData = (LPSTR)MapViewOfFile(hFileMap,FILE_MAP_ALL_ACCESS,0,0,0);
		if (mData == NULL)
		{
			li->mKeep = 0;
			if (msn) msn->Release();
			if (MSNMsgr) MSNMsgr->Release();
			if (pHost) pHost->Release();
			if (!COMInitialized) // Only unload COM if is loaded by the dll
				CoUninitialize();
			CloseHandle(hFileMap);
			return;
		}

		li->mKeep = 1;	// Keep the dll loaded
		mirc_hwnd=li->mHwnd; 
	}

//----------------------------------------------------------------------------
// LoadDll -- Frees the variables and objects
//----------------------------------------------------------------------------

	int WINAPI UnloadDll(int timeout)	
	{
		if (timeout != 1) {  // Only unload the dll if /dll -u is used
			UnmapViewOfFile(mData);
			CloseHandle(hFileMap);
			if (msn) msn->Release(); 
			if (MSNMsgr) MSNMsgr->Release();
			if (pHost) pHost->Release();
			if (!COMInitialized) // Only unload COM if is loaded by the dll
				CoUninitialize();
			return 1;
		}
		return 0;
	}

//----------------------------------------------------------------------------
// DLLInfo -- Returns version information
//----------------------------------------------------------------------------

	int WINAPI DLLInfo (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		lstrcpyn(data,"msn.dll 2.0 (c) Ivan Mendez",900); 
		return 3;
	}

//----------------------------------------------------------------------------
// AddContact -- Launches the Add a Contact wizard to the screen used for 
// adding a contact, as designated by the input parameter.
// 
// Parameters
//	The contact's sign in name
//
// Notes
//	If SigninName isn't suplied the Add a Contact wizard remains displayed 
//	for the user to input the value.
//----------------------------------------------------------------------------

	int WINAPI AddContact (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		HRESULT res = msn->raw_AddContact(0,str);
		SysFreeString(str);
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// AutoSignin -- Logs the local client back on automatically using the last
// sign-in name and saved password information.
//----------------------------------------------------------------------------

	int WINAPI AutoSignin (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		HRESULT res = msn->raw_AutoSignin();
		lstrcpyn(data,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// CreateGroup -- Creates a new group.
//
// Parameters 
//	The name of the group
//----------------------------------------------------------------------------

	int WINAPI CreateGroup (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		BSTR sid = NULL;
		IDispatch *Group;
		VARIANT service;
		char_to_bstr(data,str);
		HRESULT res = msn->get_MyServiceId(&sid);
		if (FAILED(res))
		{
			SysFreeString(str);
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		VariantInit(&service);
		service.vt = VT_BSTR;
		service.bstrVal  = sid;
		res = msn->raw_CreateGroup(str,service,&Group);
		if (FAILED(res))
		{
			SysFreeString(str);
			SysFreeString(sid);
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		SysFreeString(str);
		SysFreeString(sid);
		lstrcpyn(data,Result(res),900);
		if (Group) Group->Release();
		return 3;
	}

//----------------------------------------------------------------------------
// get_ContactsSortOrder -- Retrieves the contact current sort order for the 
// local client, which is used to determine if the contacts should be sorted 
// by groups or by their online/offline status.
//----------------------------------------------------------------------------

	int WINAPI get_ContactsSortOrder (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		MUASORT sort;
		HRESULT res = msn->get_ContactsSortOrder(&sort);
		if (FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		lstrcpyn(data,(!sort ? "Groups" : "Online/Offline"),900);
		return 3;
	}

//----------------------------------------------------------------------------
// put_ContactsSortOrder -- Sets the contact sort order for the local client, 
// which is used to determine if the  contacts should be sorted by groups or 
// by their online/offline status.
//
// Parameters
//	0 - Sorts the local user's contacts by groups.
//	1 - Sorts the local user's contacts by online/offline status.
//----------------------------------------------------------------------------

	int WINAPI put_ContactsSortOrder (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		HRESULT res = msn->put_ContactsSortOrder(static_cast<MUASORT>(atoi(data)));
		lstrcpyn(data,Result(res),900);
		return 3;
	}


//----------------------------------------------------------------------------
// get_MyFriendlyName -- Retrieves the friendly (display) name of the local
// client user.
//----------------------------------------------------------------------------

	int WINAPI get_MyFriendlyName (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		HRESULT res = msn->get_MyFriendlyName(&str);
		if (FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		bstr_to_char(str,data,900);
		SysFreeString(str);
		return 3; 
	}

//----------------------------------------------------------------------------
// get_MyPhoneNumber -- Retrieves the stored phone number of the local 
// client user.
//
// Parameters
//		0 - Get home phone number (default).
//		1 - Get work phone number.
//		2 - Get mobile phone number.
//----------------------------------------------------------------------------

	int WINAPI get_MyPhoneNumber (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		HRESULT res = msn->get_MyPhoneNumber(static_cast<MPHONE_TYPE>(atoi(data))
			,&str);
		if (FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		bstr_to_char(str,data,900);
		SysFreeString(str);
		return 3;
	}

//----------------------------------------------------------------------------
// get_MyServiceId -- Retrieves the service ID, a globally unique identifier 
// (GUID), of the primary service to which the local client user is currently 
// signed in.
//----------------------------------------------------------------------------

	int WINAPI get_MyServiceId (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		HRESULT res = msn->get_MyServiceId(&str);
		if (FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		bstr_to_char(str,data,900);
		SysFreeString(str);
		return 3;
	}

//----------------------------------------------------------------------------
// get_MyServiceName -- Retrieves the service name of the primary service 
// to which the local client user is currently signed in.
//----------------------------------------------------------------------------

	int WINAPI get_MyServiceName (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		HRESULT res = msn->get_MyServiceName(&str);
		if (FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		bstr_to_char(str,data,900);
		SysFreeString(str);
		return 3;
	}

//----------------------------------------------------------------------------
// get_MySigninName -- Retrieves the sign-in name of the local client user.
//----------------------------------------------------------------------------

	int WINAPI get_MySigninName (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		HRESULT res = msn->get_MySigninName(&str);
		if (FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		bstr_to_char(str,data,900);
		SysFreeString(str);
		return 3;
	}

//----------------------------------------------------------------------------
// get_MyStatus -- Retrieves the connection state of the local client user.
//----------------------------------------------------------------------------

	int WINAPI get_MyStatus (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		MISTATUS status;
		HRESULT res = msn->get_MyStatus(&status);
		if (FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		lstrcpyn(data,Status(status),900);
		return 3;
	}


//----------------------------------------------------------------------------
// put_MyStatus -- Sets the connection state of the local client user.

// Parameters
//	0 - Online
//	1 - Invisible
//	2 - Busy
//	3 - Be right back
//	4 - Away
//	5 - On the phone
//	6 - Out to lunch
//----------------------------------------------------------------------------

	int WINAPI put_MyStatus (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		MISTATUS mStatus;
		switch (atoi(data)) {
			case 0:
				mStatus = MISTATUS_ONLINE;
				break;
			case 1:
				mStatus = MISTATUS_INVISIBLE;
				break;
			case 2:
				mStatus = MISTATUS_BUSY;
				break;
			case 3:
				mStatus = MISTATUS_BE_RIGHT_BACK;
				break;
			case 4:
				mStatus = MISTATUS_AWAY;
				break;
			case 5:
				mStatus = MISTATUS_ON_THE_PHONE;
				break;
			case 6:
				mStatus = MISTATUS_OUT_TO_LUNCH;
				break;
			default:
				mStatus = MISTATUS_ONLINE;
				break;
		}
		HRESULT res = msn->put_MyStatus(mStatus);
		lstrcpyn(data,Result(res),900);
		return 3;
	}


//----------------------------------------------------------------------------
// get_UnreadEmailCount -- Retrieves the path to the directory currently being used 
// to store any files received.
//----------------------------------------------------------------------------

	int WINAPI get_UnreadEmailCount (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		long count;
		HRESULT res = msn->get_UnreadEmailCount(MUAFOLDER_INBOX,&count);
		if (FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		_snprintf_s(data,900,_TRUNCATE,"%u",count);
		return 3;
	}

//----------------------------------------------------------------------------
// get_ReceiveFileDirectory -- Retrieves the path to the directory currently being used 
// to store any files received.
//----------------------------------------------------------------------------

	int WINAPI get_ReceiveFileDirectory (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		HRESULT res = msn->get_ReceiveFileDirectory(&str);
		if (FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		bstr_to_char(str,data,900);
		SysFreeString(str);
		return 3;
	}

//----------------------------------------------------------------------------
// InstantMessage -- Launches a conversation window with the recipient.
//
// Parameters
//	Recipient's SigninName
//----------------------------------------------------------------------------

	int WINAPI InstantMessage (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		BSTR str = NULL;
		VARIANT signinname;
		IDispatch *twindow;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		VariantInit(&signinname);
		signinname.vt = VT_BSTR;
		signinname.bstrVal  = str;
		HRESULT res = msn->raw_InstantMessage(signinname,&twindow);
		SysFreeString(str);
		if (twindow) twindow->Release();
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// InviteApp -- Issues an invitation to a user to use a specified application.
//
// Parameters
//	Contact
//	Application GUID
//----------------------------------------------------------------------------

	int WINAPI InviteApp (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		BSTR str = NULL;
		BSTR guid = NULL;
		IDispatch *twindow;
		char* tmp = data;
		VARIANT contact;
		char_to_bstr(getword(data),str);
		char_to_bstr(data,guid);
		VariantInit(&contact);
		contact.vt = VT_BSTR;
		contact.bstrVal  = str;
		HRESULT res = msn->raw_InviteApp(contact,guid,&twindow);
		SysFreeString(str);
		SysFreeString(guid);
		if (twindow) twindow->Release();
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// MediaWizard -- Launches the Audio and Video Tuning Wizard dialog box.
//----------------------------------------------------------------------------

	int WINAPI MediaWizard (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		HRESULT res = msn->raw_MediaWizard(0);
		lstrcpyn(data,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// OpenInbox -- Launches the default e-mail application and opens the Inbox.
//----------------------------------------------------------------------------

	int WINAPI OpenInbox (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		HRESULT res = msn->raw_OpenInbox();
		lstrcpyn(data,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// OptionsPages -- Launches the Options dialog box with a specified tab 
// initially open.
//
// Parameters
//	0 - Display the General tab of the Options dialog box (default).
//	1 - Display the Privacy tab of the Options dialog box.
//	2 - Display the Exchange tab of the Options dialog box.
//	3 - Display the Accounts tab of the Options dialog box.
//	4 - Display the Connection tab of the Options dialog box.
//	5 - Display the Preferences tab of the Options dialog box.
//	6 - Display the Services tab of the Options dialog box.
//	7 - Display the Phone tab of the Options dialog.
//----------------------------------------------------------------------------

	int WINAPI OptionsPages (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		HRESULT res = msn->raw_OptionsPages(0,static_cast<MOPTIONPAGE>(atoi(data)));
		lstrcpyn(data,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// Page --  Launches the Page window that is used to page a user, as 
//	designated by the input parameter.
//
// Parameters
//	Recipient's SigninName
//----------------------------------------------------------------------------

	int WINAPI Page (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		BSTR str = NULL;
		VARIANT contact;
		IDispatch *twindow;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		VariantInit(&contact);
		contact.vt = VT_BSTR;
		contact.bstrVal  = str;
		HRESULT res = msn->raw_Page(contact,&twindow);
		SysFreeString(str);
		if (twindow) twindow->Release();
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// Phone --  Launches the Phone Call dialog box. 
//
// Parameters
//	Recipient's SigninName
//	Phone type
//	Phone number
//----------------------------------------------------------------------------

	int WINAPI Phone (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		BSTR str = NULL;
		BSTR phone = NULL;
		VARIANT contact;
		IDispatch *pwindow;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		MPHONE_TYPE type = static_cast<MPHONE_TYPE>(atoi(getword(data)));
		char_to_bstr(data,phone);
		VariantInit(&contact);
		contact.vt = VT_BSTR;
		contact.bstrVal  = str;
		HRESULT res = msn->raw_Phone(contact,type,phone,&pwindow);
		SysFreeString(str);
		SysFreeString(phone);
		if (pwindow) pwindow->Release(); 
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// SendFile --  Launches the Send File mode of a conversation window to a
// specified contact. 
//
// Parameters
//	Recipient's SigninName
//	File name (optional)
//
// Notes
//	Returns E_NOTIMPL?
//----------------------------------------------------------------------------

	int WINAPI SendFile (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		BSTR str = NULL;
		BSTR filename = NULL;
		VARIANT contact;
		IDispatch *pwindow;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		char_to_bstr(data,filename);
		VariantInit(&contact);
		contact.vt = VT_BSTR;
		contact.bstrVal  = str;
		HRESULT res = msn->raw_SendFile(contact,filename,&pwindow);
		SysFreeString(str);
		SysFreeString(filename);
		if (pwindow) pwindow->Release(); 
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// SendMail --  Launches a compose window in the client's default e-mail 
// application with the To: line prepopulated with the sign-in name of the
// specified contact.
//
// Parameters
//	Recipient's e-mail
//----------------------------------------------------------------------------

	int WINAPI SendMail (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		BSTR str = NULL;
		VARIANT contact;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		VariantInit(&contact);
		contact.vt = VT_BSTR;
		contact.bstrVal  = str;
		HRESULT res = msn->raw_SendMail(contact);
		SysFreeString(str);
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// Signin --  Launches the Messenger client Sign In dialog box and populates 
// the sign-in name field.
//
// Parameters
//	Signin name (optional)
//	Password (optional)
//----------------------------------------------------------------------------

	int WINAPI Signin (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		BSTR password = NULL;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		char_to_bstr(getword(data),password);
		HRESULT res = msn->raw_Signin(0,str,password);
		SysFreeString(str);
		SysFreeString(password);
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// Signout -- Signs the current client user out of all Messenger services.
//----------------------------------------------------------------------------

	int WINAPI Signout (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		HRESULT res = msn->raw_Signout();
		lstrcpyn(data,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// StartVideo --  Launches a conversation window to initiate a video session
// with a particular contact, pending acceptance of the invitation.
//
// Parameters
//	Contact
//----------------------------------------------------------------------------

	int WINAPI StartVideo (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		BSTR str = NULL;
		IDispatch *pwindow;
		VARIANT contact;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		VariantInit(&contact);
		contact.vt = VT_BSTR;
		contact.bstrVal  = str;
		HRESULT res = msn->raw_StartVideo(contact,&pwindow);
		SysFreeString(str);
		if (pwindow) pwindow->Release(); 
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}


//----------------------------------------------------------------------------
// StartVoice --  Launches a Messenger conversation window to initiate a voice
// message session with a particular contact, pending acceptance of the 
// invitation. 
//
// Parameters
//	Contact
//----------------------------------------------------------------------------

	int WINAPI StartVoice (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		BSTR str = NULL;
		IDispatch *pwindow;
		VARIANT contact;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		VariantInit(&contact);
		contact.vt = VT_BSTR;
		contact.bstrVal  = str;
		HRESULT res = msn->raw_StartVoice(contact,&pwindow);
		SysFreeString(str);
		if (pwindow) pwindow->Release(); 
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// ViewProfile --  Launches a new Web browser instance, allowing the client 
// user to view properties of the specified contact through the Public 
// Profiles feature.
//
// Parameters
//	Contact
//----------------------------------------------------------------------------

	int WINAPI ViewProfile (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		BSTR str = NULL;
		VARIANT contact;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		VariantInit(&contact);
		contact.vt = VT_BSTR;
		contact.bstrVal  = str;
		HRESULT res = msn->raw_ViewProfile(contact);
		SysFreeString(str);
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// put_MyDisplayPicture --  Changes the local client display picture.
//
// Parameters
//	Path to the new picture.
//----------------------------------------------------------------------------

	int WINAPI put_MyDisplayPicture (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		BSTR str = NULL;
		VARIANT path;
		char_to_bstr(data,str);
		VariantInit(&path);
		path.vt = VT_BSTR;
		path.bstrVal  = str;
		HRESULT res = msn->put_MyProperty(MCONTACTPROP_USERTILE_PATH,path);
		SysFreeString(str);
		lstrcpyn(data,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// put_MyPSM --  Changes the local Personal Status Message.
//
// Parameters
//	Personal Status Message
//----------------------------------------------------------------------------

	int WINAPI put_MyPSM (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!MSNMsgr)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		BSTR str = NULL;
		VARIANT psm;
		char_to_bstr(data,str);
		VariantInit(&psm);
		psm.vt = VT_BSTR;
		psm.bstrVal  = str;
		HRESULT res = MSNMsgr->SetLocalProperty(static_cast<MUSERPROPERTY>(34),psm);
		SysFreeString(str);
		lstrcpyn(data,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// put_MyFriendlyName --  Changes the local Friendly Name.
//
// Parameters
//	Friendly Name
//----------------------------------------------------------------------------

	int WINAPI put_MyFriendlyName (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!MSNMsgr)
			return 0;
		if (!lstrlen(data))
		{
			lstrcpyn(data,Result(E_INVALIDARG),900);
			return 3;
		}
		IMSNMsgrServices *services = NULL;
		IMSNMsgrService *pservice = NULL;
		BSTR str = NULL;
		VARIANT psm;
		char_to_bstr(data,str);
		VariantInit(&psm);
		psm.vt = VT_BSTR;
		psm.bstrVal  = str;
		HRESULT res= MSNMsgr->GetServices(&services);
		if (FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		res = services->GetPrimaryService(&pservice);
		if (FAILED(res))
		{
			if (services) services->Release();
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		res = pservice->SetFriendlyName(str);
		if (FAILED(res))
		{
			if (services) services->Release();
			if (pservice) pservice->Release();
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		SysFreeString(str);
		if (services) services->Release();
		if (pservice) pservice->Release();
		lstrcpyn(data,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// get_Blocked -- Retrieves a Boolean value declaring whether the contact 
// associated with this SigninName is blocked by the current client user.
//
// Parameters
//	The contact's SigninName
//----------------------------------------------------------------------------

	int WINAPI get_Blocked (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		VARIANT_BOOL blocked;
		char_to_bstr(data,str);
		HRESULT res = GetContact(str);
		SysFreeString(str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		res = IMcontact->get_Blocked(&blocked);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		if (IMcontact) IMcontact->Release();
		_snprintf_s(data,900,_TRUNCATE,"%u",(blocked == VARIANT_TRUE ? 1 : 0));
		return 3;
	}

//----------------------------------------------------------------------------
// put_Blocked -- Gets a Boolean value declaring whether the contact associated 
// with this SigninName is blocked by the current client user.
//
// Parameters
//	Contact
//	Boolean blocked
//----------------------------------------------------------------------------

	int WINAPI put_Blocked (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		char *tmp = data;
		BSTR str = NULL;
		char_to_bstr(getword(data),str);
		VARIANT_BOOL blocked = static_cast<VARIANT_BOOL>(atoi(data));
		HRESULT res = GetContact(str);
		SysFreeString(str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		res = IMcontact->put_Blocked(blocked);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		if (IMcontact) IMcontact->Release();
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}

//----------------------------------------------------------------------------
// get_CanPage -- Retrieves a Boolean value declaring whether the contact 
// associated with this SigninName has mobile preferences established or 
// visible.
//
// Parameters
//	The contact's SigninName
//----------------------------------------------------------------------------

	int WINAPI get_CanPage (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		VARIANT_BOOL page;
		char_to_bstr(data,str);
		HRESULT res = GetContact(str);
		SysFreeString(str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		res = IMcontact->get_CanPage(&page);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		if (IMcontact) IMcontact->Release();
		_snprintf_s(data,900,_TRUNCATE,"%u",(page == VARIANT_TRUE ? 1 : 0));
		return 3;
	}

//----------------------------------------------------------------------------
// get_FriendlyName -- Retrieves the friendly name of the contact associated with
// this SigninName.
//
// Parameters
//	The contact's SigninName
//----------------------------------------------------------------------------

	int WINAPI get_FriendlyName (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		char_to_bstr(data,str);
		HRESULT res = GetContact(str);
		SysFreeString(str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		res = IMcontact->get_FriendlyName(&str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		bstr_to_char(str,data,900);
		SysFreeString(str);
		if (IMcontact) IMcontact->Release();
		return 3;
	}

//----------------------------------------------------------------------------
// get_IsSelf -- Retrieves a Boolean value declaring whether the contact 
// associated with this SigninName is actually the current client user.
//
// Parameters
//	The contact's SigninName
//----------------------------------------------------------------------------

	int WINAPI get_IsSelf (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		VARIANT_BOOL self;
		char_to_bstr(data,str);
		HRESULT res = GetContact(str);
		SysFreeString(str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		res = IMcontact->get_IsSelf(&self);
		if (FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		if (IMcontact) IMcontact->Release();
		_snprintf_s(data,900,_TRUNCATE,"%u",(self == VARIANT_TRUE ? 1 : 0));
		return 3;
	}

//----------------------------------------------------------------------------
// get_PhoneNumber -- Retrieves the friendly name of the contact associated with
// this SigninName.
//
// Parameters
//	The contact's SigninName
//	Phone type
//----------------------------------------------------------------------------

	int WINAPI get_PhoneNumber (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		MPHONE_TYPE PhoneType= static_cast<MPHONE_TYPE>(atoi(data));
		HRESULT res = GetContact(str);
		SysFreeString(str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		res = IMcontact->get_PhoneNumber(PhoneType, &str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		bstr_to_char(str,tmp,900);
		SysFreeString(str);
		if (IMcontact) IMcontact->Release();
		return 3;
	}

//----------------------------------------------------------------------------
// get_ServiceId -- Retrieves the service ID, a globally unique identifier 
// (GUID), for the contact associated with this SigninName.
//
// Parameters
//	The contact's SigninName
//----------------------------------------------------------------------------

	int WINAPI get_ServiceId (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		HRESULT res = GetContact(str);
		SysFreeString(str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		res = IMcontact->get_ServiceId(&str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		bstr_to_char(str,tmp,900);
		SysFreeString(str);
		if (IMcontact) IMcontact->Release();
		return 3;
	}

//----------------------------------------------------------------------------
// get_ServiceName -- Retrieves the service name of the contact associated 
// with this SigninName.
//
// Parameters
//	The contact's SigninName
//----------------------------------------------------------------------------

	int WINAPI get_ServiceName (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		HRESULT res = GetContact(str);
		SysFreeString(str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		res = IMcontact->get_ServiceName(&str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		bstr_to_char(str,tmp,900);
		SysFreeString(str);
		if (IMcontact) IMcontact->Release();
		return 3;
	}

//----------------------------------------------------------------------------
// get_Status -- Retrieves the service name of the contact associated 
// with this SigninName.
//
// Parameters
//	The contact's SigninName
//----------------------------------------------------------------------------

	int WINAPI get_Status (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		char *tmp = data;
		MISTATUS status;
		char_to_bstr(getword(data),str);
		HRESULT res = GetContact(str);
		SysFreeString(str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		res = IMcontact->get_Status(&status);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		if (IMcontact) IMcontact->Release();
		lstrcpyn(tmp,Status(status),900);
		return 3;
	}

//----------------------------------------------------------------------------
// get_Count -- Retrieves the number of Contacts/Groups.
//
// Parameters
//	Contacts/Groups
//----------------------------------------------------------------------------

	int WINAPI get_Count (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		LONG pcCount;
		HRESULT res;

		switch (*data)
		{
			default:
			case 'c':
			case 'C':
				res = get_MyContacts();
				if(FAILED(res))
				{
					lstrcpyn(data,Result(res),900);
					return 3;
				}
				res = pContacts->get_Count(&pcCount);
				pContacts->Release();
				break;
			case 'g':
			case 'G':
				res = get_MyGroups();
				if(FAILED(res))
				{
					lstrcpyn(data,Result(res),900);
					return 3;
				}
				res = pGroups->get_Count(&pcCount);
				pGroups->Release();
				break;
		}
		if(FAILED(res))
		{
			lstrcpyn(data,Result(res),900);
			return 3;
		}
		_snprintf_s(data,900,_TRUNCATE,"%u",pcCount);
		return 3;
	}

//----------------------------------------------------------------------------
// Item -- Retrieves a specific SigninName/GroupName by numeric index.
//
// Parameters
//	Index
//	Contacts/Groups
//----------------------------------------------------------------------------

	int WINAPI Item (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		HRESULT res;
		char *tmp = data;
		IDispatch *pobj;
		long index = atol(getword(data));

		switch (*data)
		{
			default:
			case 'c':
			case 'C':
				res = get_MyContacts();
				if(FAILED(res))
				{
					lstrcpyn(tmp,Result(res),900);
					return 3;
				}
				res = pContacts->raw_Item(index,&pobj);
				if(FAILED(res))
				{
					if (pContacts) pContacts->Release();
					lstrcpyn(tmp,Result(res),900);
					return 3;
				}
				res = pobj->QueryInterface(__uuidof(IMessengerContact),reinterpret_cast<void**>(&IMcontact));
				if(FAILED(res))
				{
					if (pContacts) pContacts->Release();
					if (pobj) pobj->Release();
					lstrcpyn(tmp,Result(res),900);
					return 3;
				}
				res = IMcontact->get_SigninName(&str);
				if (pContacts) pContacts->Release();
				if (pobj) pobj->Release();
				if (IMcontact) IMcontact->Release();
				break;
			case 'g':
			case 'G':
				res = get_MyGroups();
				if(FAILED(res))
				{
					lstrcpyn(tmp,Result(res),900);
					return 3;
				}
				res = pGroups->raw_Item(index,&pobj);
				if(FAILED(res))
				{
					if (pGroups) pGroups->Release();
					lstrcpyn(tmp,Result(res),900);
					return 3;
				}
				res = pobj->QueryInterface(__uuidof(IMessengerGroup),reinterpret_cast<void**>(&IMgroup));
				if(FAILED(res))
				{
					if (pGroups) pGroups->Release();
					if (pobj) pobj->Release();
					lstrcpyn(tmp,Result(res),900);
					return 3;
				}
				res = IMgroup->get_Name(&str);
				if (IMgroup) IMgroup->Release();
				if (pobj) pobj->Release();
				if (IMcontact) IMcontact->Release();
				break;
		}
		if(FAILED(res))
		{
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		bstr_to_char(str,tmp,900);
		SysFreeString(str);
		return 3;
	}

//----------------------------------------------------------------------------
// Remove -- Removes a contact. 
//
// Parameters
//	SigninName
//----------------------------------------------------------------------------

	int WINAPI Remove (HWND mWnd, HWND aWnd, char *data, 
		char *parms, BOOL show, BOOL nopause)
	{
		if (!msn)
			return 0;
		BSTR str = NULL;
		char *tmp = data;
		char_to_bstr(getword(data),str);
		HRESULT res = GetContact(str);
		SysFreeString(str);
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		res = get_MyContacts();
		if (FAILED(res))
		{
			if (IMcontact) IMcontact->Release();
			lstrcpyn(tmp,Result(res),900);
			return 3;
		}
		res = pContacts->Remove(IMcontact);
		if (IMcontact) IMcontact->Release();
		if (pContacts) pContacts->Release();
		lstrcpyn(tmp,Result(res),900);
		return 3;
	}
}
